#!/bin/bash
#
#    This script should run on the first of every month to automatically commit
#    the event entry for the next Fellowship meeting. This is still work in
#    progress. Additional features may be added upon request. Please report any
#    problems to the author (see below).
#
#    Script version 0.1
#
#    Copyright © 2012 by Martin Gollowitzer <gollo@fsfe.org>
#
#    Permission is hereby granted, free of charge, to any person obtaining a
#    copy of this software and associated documentation files (the "Software"),
#    to deal in the Software without restriction, including without limitation
#    the rights to use, copy, modify, merge, publish, distribute, sublicense,
#    and/or sell copies of the Software, and to permit persons to whom the
#    Software is furnished to do so, subject to the following conditions:
#
#    The above copyright notice and this permission notice shall be included in
#    all copies or substantial portions of the Software.
#
#    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#    IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#    FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#    AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
#    LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
#    FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
#     DEALINGS IN THE SOFTWARE.

### Begin configuration section ###

## General options ## 

# please enter the script's base directory
basedir="."

# Please enter day of week for the recurring event (0-6, Sunday=0)
# Friday is 5
recurrday=5

# Enter the number of the week of month for the event (0-3, 0 means "First $recurrday every month")
recurrweek=2

# Enter location for Event. Text fragements must exist in the according directory
location="Vienna"


## SVN options ##
# User name for svn
svnuser="user"

# Password for SVN
svnpassword="password"

# SVN URI
svnuri="https://svn.fsfe.org/fsfe-web/trunk"

# commit message for auto commits
# At the end the text "Commiting event in $location on <date>" is added
commitmsg="Automatic Fellowship meeting event commit script"


## Debugging options ##
# enable debug messages?
debug=0

# Text to print at the beginning of debug output lines
debugtext="DEBUG:"

### End of configuration section ###



# Set the SVN commandline
[[ "$debug" != "1" ]] && svnquiet="-q"
svncmd="svn $svnquiet --username $svnuser --password $svnpassword"

# check if the base directory exists
[ ! -d $basedir ] && echo "$basedir does not exist!" && exit 1

# Delete data from SVN completely. A new checkout is done every time the script is run
[ -d $basedir/svn ] && rm -rf $basedir/svn
mkdir $basedir/svn

# What's the current year?
thisyear=$(date +%Y)

# What's the current month?
thismonth=$(date +%m)

# Let's check out this year's events directory
[[ "$debug" == "1" ]] && echo "$debugtext Starting SVN checkout"
$svncmd co $svnuri/events/$thisyear $basedir/svn/
retcode=$?
[[ $retcode -ne 0 ]] && echo "SVN checkout failed! Return code: $retcode" && exit $retcode


# This finds the right day for the event entry

daycount=0

for s in `seq 1 28`
do
 weekday=$(date +%w --date="$thisyear-$thismonth-$s")

 if [ "$weekday" == "$recurrday" ]; then
  [[ "$debug" == "1" ]] && echo "$debugtext $thisyear-$thismonth-$s is a %RECURRDAY"
  friday[$daycount]=$s
  daycount=$((daycount+1))
 fi
done

date="$thisyear-$thismonth-${friday[$recurrweek]}"

[[ "$debug" == "1" ]] && echo "$debugtext Date for Event: $date"

# date for filename
filedate=$(date -d "$date" +%Y%m%d)
[[ "$debug" == "1" ]] && echo "$debugtext filedate: $filedate"

# date for file content
useddate=$(date -d "$date" +%Y-%m-%d)
[[ "$debug" == "1" ]] && echo "$debugtext useddate: $useddate"

# Adapting commit message:
commitmsg="$commitmsg >>> Commiting event in $location on $useddate"
[[ "$debug" == "1" ]] && echo "$debugtext Commit message is now \"$commitmsg\""

# if $go is not zero, we can safely start writing the files without 
# overwriting old stuff
go=0

# $filecount is used for the filename. If there is already an event 
# on the date specified, we increase this until we have a "free" file name
filecount=1

# walk through all the languages available for the specified location
for lang in $(ls $basedir/buildingblocks/$location/); do

  # $finished is set to 1 after the file has been written; this does not happen 
  # before we don't have a usable serial number for our event entries
  finished=0
  while [ $finished -eq 0 ]; do

    # event serial number must have two digits
    if [ $filecount -lt 10 ]; then
      newcount="0$filecount"
    else
      newcount=$filecount
    fi

    # file name for our event
    file="$basedir/svn/event-$filedate-$newcount.$lang"

    # finds all files with the name we want to use
    todaycontent=$(find $basedir/svn/ -type f | grep -v "\.svn" | grep -i event-$filedate-$newcount)

    # if the list created above is *not* empty, we try the next serial number, 
    # because there's a file with that serial number in at least one language
    # otherwise, we set $go to 1 so we know it's save to start writing files
    if [ "$todaycontent" != "" ]; then
      [ $go -eq 0 ] && filecount=$[$filecount+1]
      [[ "$debug" == "1" ]] && echo "$debugtext File serial number raised to $filecount"
    else
      go=1
    fi
    
    # only write those files if $go has been set to 1!
    if [ $go -eq 1 ]; then
      cat buildingblocks/general/head > $file
      echo "   <event start=\"$useddate\" end=\"$useddate\">" >> $file
      cat buildingblocks/$location/$lang >> $file
      cat buildingblocks/general/tail >> $file
      [[ "$debug" == "1" ]] && echo "$debugtext File $file written; no adding to svn"
      svn $svnquiet add $file 

      finished=1
    fi
  done

done

# Let's do a checkin on the files
[[ "$debug" == "1" ]] && echo "$debugtext Starting SVN checkin"
$svncmd -m "$commitmsg" ci $basedir/svn/* && echo "SVN checkin completed."

echo "Automatic event entry for event in $location on $useddate has been commited to FSFE website."
echo "Visit http://fsfe.org to check the results."

exit 0

# vim: set ft=sh nohls ai :
